package org.example.todo;

import io.swagger.v3.oas.annotations.OpenAPIDefinition;
import io.swagger.v3.oas.annotations.info.Info;
import io.swagger.v3.oas.annotations.media.Content;
import io.swagger.v3.oas.annotations.responses.ApiResponse;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.http.HttpStatus;
import org.springframework.http.ProblemDetail;
import org.springframework.validation.annotation.Validated;
import org.springframework.web.bind.annotation.DeleteMapping;
import org.springframework.web.bind.annotation.ExceptionHandler;
import org.springframework.web.bind.annotation.GetMapping;
import org.springframework.web.bind.annotation.PathVariable;
import org.springframework.web.bind.annotation.PostMapping;
import org.springframework.web.bind.annotation.PutMapping;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.ResponseStatus;
import org.springframework.web.bind.annotation.RestController;

import jakarta.validation.ConstraintViolationException;
import jakarta.validation.Valid;
import jakarta.validation.constraints.Min;
import java.util.List;

import static org.springframework.http.MediaType.APPLICATION_JSON_VALUE;
import static org.springframework.http.MediaType.APPLICATION_XML_VALUE;

@RestController
@RequestMapping("/todos")
@Validated
@OpenAPIDefinition(info = @Info(title = "Todo Service", version = "1.0"))
public class TodoListImpl {

	private static final String TEXT_CSV_VALUE = "text/csv";
	private static final Logger log = LoggerFactory.getLogger(TodoListImpl.class);

	private final TodoRepository todoRepository;

	public TodoListImpl(TodoRepository todoRepository) {
		this.todoRepository = todoRepository;
	}

	@GetMapping(produces = {APPLICATION_JSON_VALUE, APPLICATION_XML_VALUE, TEXT_CSV_VALUE})
	public List<Todo> getTodos() {
		return todoRepository.findAll();
	}

	@GetMapping(value = "{id}", produces = APPLICATION_JSON_VALUE)
	@ApiResponse(responseCode = "200", description = "todo found")
	@ApiResponse(responseCode = "404", description = "todo not found", content = @Content)
	public Todo findTodo(@PathVariable @Min(1) long id) throws TodoNotFoundException {
		return todoRepository.findById(id)
				.orElseThrow(() -> new TodoNotFoundException("Todo with id " + id + " not found"));
	}

	@PostMapping(consumes = APPLICATION_JSON_VALUE, produces = APPLICATION_JSON_VALUE)
	@ResponseStatus(HttpStatus.CREATED)
	@ApiResponse(responseCode = "201", description = "todo added")
	public long addTodo(@RequestBody @Valid Todo todo) {
		if (todo.getId() != null)
			throw new IllegalArgumentException("Todo must not have an id");
		todo = todoRepository.save(todo);
		log.info("Todo with id " + todo.getId() + " added");
		return todo.getId();
	}

	@PutMapping(value = "{id}", consumes = APPLICATION_JSON_VALUE)
	@ResponseStatus(HttpStatus.NO_CONTENT)
	@ApiResponse(responseCode = "204", description = "todo updated")
	@ApiResponse(responseCode = "400", description = "invalid todo", content = @Content)
	@ApiResponse(responseCode = "404", description = "todo not found", content = @Content)
	public void updateTodo(@PathVariable @Min(1) long id, @RequestBody @Valid Todo todo) throws TodoNotFoundException {
		if (todo.getId() == null || todo.getId() != id)
			throw new IllegalArgumentException("Todo has an invalid id");
		if (!todoRepository.existsById(id))
			throw new TodoNotFoundException("Todo with id " + id + " not found");
		todoRepository.save(todo);
		log.info("Todo with id " + todo.getId() + " updated");
	}

	@DeleteMapping("{id}")
	@ResponseStatus(HttpStatus.NO_CONTENT)
	@ApiResponse(responseCode = "204", description = "todo deleted")
	public void removeTodo(@PathVariable @Min(1) long id) {
		todoRepository.findById(id).ifPresent(todoRepository::delete);
		log.info("Todo with id " + id + " removed");
	}

	@ExceptionHandler
	public ProblemDetail handle(IllegalArgumentException ex) {
		return ProblemDetail.forStatusAndDetail(HttpStatus.BAD_REQUEST, ex.getMessage());
	}

	@ExceptionHandler
	public ProblemDetail handle(ConstraintViolationException ex) {
		return ProblemDetail.forStatusAndDetail(HttpStatus.BAD_REQUEST, ex.getMessage());
	}

	@ExceptionHandler
	public ProblemDetail handle(TodoNotFoundException ex) {
		return ProblemDetail.forStatusAndDetail(HttpStatus.NOT_FOUND, ex.getMessage());
	}
}
