package org.example.todo;

import io.restassured.RestAssured;
import io.restassured.http.ContentType;
import org.junit.jupiter.api.BeforeEach;
import org.junit.jupiter.api.Test;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.boot.test.context.SpringBootTest;
import org.springframework.boot.test.web.server.LocalServerPort;

import java.util.Date;

import static org.hamcrest.Matchers.empty;
import static org.hamcrest.Matchers.is;
import static org.hamcrest.Matchers.not;
import static org.springframework.boot.test.context.SpringBootTest.WebEnvironment;
import static org.springframework.http.HttpStatus.CREATED;
import static org.springframework.http.HttpStatus.NO_CONTENT;
import static org.springframework.http.HttpStatus.OK;
import static org.springframework.http.MediaType.APPLICATION_JSON_VALUE;

@SpringBootTest(webEnvironment = WebEnvironment.RANDOM_PORT)
public class TodoRestAssuredIT {

	private static final String BASE_PATH = "/todos";

	@LocalServerPort
	private int port;
	@Value("${server.servlet.contextPath}")
	private String contextPath;

	@BeforeEach
	public void init() {
		RestAssured.port = port;
		RestAssured.basePath = contextPath;
	}

	@Test
	public void addTodo() {
		Todo todo = new Todo("Test", new Date());
		RestAssured.given().contentType(ContentType.JSON).body(todo)
				.accept(ContentType.JSON)
				.when().post(BASE_PATH)
				.then().statusCode(CREATED.value())
				.assertThat().contentType(APPLICATION_JSON_VALUE);
	}

	@Test
	public void getTodos() {
		RestAssured.given().accept(ContentType.JSON)
				.when().get(BASE_PATH)
				.then().statusCode(OK.value())
				.assertThat().contentType(APPLICATION_JSON_VALUE)
				.body("$", not(empty()));
	}

	@Test
	public void findTodo() {
		RestAssured.given().accept(ContentType.JSON)
				.when().get(BASE_PATH + "/" + 1)
				.then().statusCode(OK.value())
				.assertThat().contentType(APPLICATION_JSON_VALUE)
				.body("id", is(1));
	}

	@Test
	public void completeTodo() {
		Todo todo = new Todo(2, "Test", new Date());
		todo.setCompleted(true);
		RestAssured.given().contentType(ContentType.JSON).body(todo)
				.when().put(BASE_PATH + "/" + 2)
				.then().statusCode(NO_CONTENT.value());
	}

	@Test
	public void deleteTodo() {
		RestAssured.given().when().delete(BASE_PATH + "/" + 3)
				.then().statusCode(NO_CONTENT.value());
	}
}
