package org.example.todo;

import org.junit.jupiter.api.Test;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.boot.test.context.SpringBootTest;
import org.springframework.boot.test.web.client.TestRestTemplate;
import org.springframework.http.HttpStatus;
import org.springframework.http.RequestEntity;
import org.springframework.http.ResponseEntity;

import java.util.Date;

import static org.assertj.core.api.Assertions.assertThat;
import static org.springframework.boot.test.context.SpringBootTest.WebEnvironment;
import static org.springframework.http.MediaType.APPLICATION_JSON;
import static org.springframework.http.RequestEntity.delete;
import static org.springframework.http.RequestEntity.get;
import static org.springframework.http.RequestEntity.post;
import static org.springframework.http.RequestEntity.put;

@SpringBootTest(webEnvironment = WebEnvironment.RANDOM_PORT)
public class TodoRestTemplateIT {

	private static final String BASE_PATH = "/todos";

	@Autowired
	private TestRestTemplate restTemplate;

	@Test
	public void addTodo() {
		Todo todo = new Todo("Test", new Date());
		RequestEntity<Todo> request = post(BASE_PATH).body(todo);
		ResponseEntity<Long> response = restTemplate.exchange(request, Long.class);
		assertThat(response.getStatusCode()).isEqualTo(HttpStatus.CREATED);
		assertThat(response.getBody()).isPositive();
	}

	@Test
	public void getTodos() {
		RequestEntity<Void> request = get(BASE_PATH).build();
		ResponseEntity<Todo[]> response = restTemplate.exchange(request, Todo[].class);
		assertThat(response.getStatusCode()).isEqualTo(HttpStatus.OK);
		assertThat(response.getHeaders().getContentType()).isEqualTo(APPLICATION_JSON);
		assertThat(response.getBody()).isNotEmpty();
	}

	@Test
	public void findTodo() {
		ResponseEntity<Todo> response = restTemplate.getForEntity(BASE_PATH + "/" + 1, Todo.class);
		assertThat(response.getStatusCode()).isEqualTo(HttpStatus.OK);
		assertThat(response.getHeaders().getContentType()).isEqualTo(APPLICATION_JSON);
		assertThat(response.getBody().getId()).isEqualTo(1);
	}

	@Test
	public void completeTodo() {
		Todo todo = new Todo(2, "Test", new Date());
		todo.setCompleted(true);
		RequestEntity<Todo> request = put(BASE_PATH + "/" + 2).body(todo);
		ResponseEntity<Void> response = restTemplate.exchange(request, Void.class);
		assertThat(response.getStatusCode()).isEqualTo(HttpStatus.NO_CONTENT);
	}

	@Test
	public void deleteTodo() {
		RequestEntity<Void> request = delete(BASE_PATH + "/" + 3).build();
		ResponseEntity<Void> response = restTemplate.exchange(request, Void.class);
		assertThat(response.getStatusCode()).isEqualTo(HttpStatus.NO_CONTENT);
	}
}
