package org.example.todo;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.http.HttpStatus;
import org.springframework.http.ProblemDetail;
import org.springframework.web.bind.annotation.DeleteMapping;
import org.springframework.web.bind.annotation.ExceptionHandler;
import org.springframework.web.bind.annotation.GetMapping;
import org.springframework.web.bind.annotation.PathVariable;
import org.springframework.web.bind.annotation.PostMapping;
import org.springframework.web.bind.annotation.PutMapping;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.ResponseStatus;
import org.springframework.web.bind.annotation.RestController;

import java.util.List;

import static org.springframework.http.MediaType.APPLICATION_JSON_VALUE;

@RestController
@RequestMapping("/todos")
public class TodoListImpl {

	private static final Logger log = LoggerFactory.getLogger(TodoListImpl.class);

	private final TodoRepository todoRepository;

	public TodoListImpl(TodoRepository todoRepository) {
		this.todoRepository = todoRepository;
	}

	@GetMapping(produces = APPLICATION_JSON_VALUE)
	public List<Todo> getTodos() {
		return todoRepository.findAll();
	}

	@GetMapping(value = "{id}", produces = APPLICATION_JSON_VALUE)
	public Todo findTodo(@PathVariable long id) throws TodoNotFoundException {
		return todoRepository.findById(id)
				.orElseThrow(() -> new TodoNotFoundException("Todo with id " + id + " not found"));
	}

	@PostMapping(consumes = APPLICATION_JSON_VALUE, produces = APPLICATION_JSON_VALUE)
	@ResponseStatus(HttpStatus.CREATED)
	public long addTodo(@RequestBody Todo todo) {
		if (todo.getId() != null)
			throw new IllegalArgumentException("Todo must not have an id");
		todo = todoRepository.save(todo);
		log.info("Todo with id " + todo.getId() + " added");
		return todo.getId();
	}

	@PutMapping(value = "{id}", consumes = APPLICATION_JSON_VALUE)
	@ResponseStatus(HttpStatus.NO_CONTENT)
	public void updateTodo(@PathVariable long id, @RequestBody Todo todo) throws TodoNotFoundException {
		if (todo.getId() == null || todo.getId() != id)
			throw new IllegalArgumentException("Todo has an invalid id");
		if (!todoRepository.existsById(id))
			throw new TodoNotFoundException("Todo with id " + id + " not found");
		todoRepository.save(todo);
		log.info("Todo with id " + todo.getId() + " updated");
	}

	@DeleteMapping("{id}")
	@ResponseStatus(HttpStatus.NO_CONTENT)
	public void removeTodo(@PathVariable long id) {
		todoRepository.findById(id).ifPresent(todoRepository::delete);
		log.info("Todo with id " + id + " removed");
	}

	@ExceptionHandler
	public ProblemDetail handle(IllegalArgumentException ex) {
		return ProblemDetail.forStatusAndDetail(HttpStatus.BAD_REQUEST, ex.getMessage());
	}

	@ExceptionHandler
	public ProblemDetail handle(TodoNotFoundException ex) {
		return ProblemDetail.forStatusAndDetail(HttpStatus.NOT_FOUND, ex.getMessage());
	}
}
