package org.example.todo;

import org.apache.cxf.annotations.SchemaValidation;
import org.apache.cxf.interceptor.InInterceptors;
import org.apache.cxf.interceptor.OutInterceptors;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.stereotype.Component;

import jakarta.jws.WebService;
import javax.xml.datatype.XMLGregorianCalendar;
import java.util.List;

@Component
@SchemaValidation
@WebService(endpointInterface = "org.example.todo.TodoList", wsdlLocation = "classpath:TodoListService.wsdl")
@InInterceptors(interceptors = "org.apache.cxf.ext.logging.LoggingInInterceptor")
@OutInterceptors(interceptors = "org.apache.cxf.ext.logging.LoggingOutInterceptor")
public class TodoListImpl implements TodoList {

	private static final Logger log = LoggerFactory.getLogger(TodoListImpl.class);

	private final TodoRepository todoRepository;

	public TodoListImpl(TodoRepository todoRepository) {
		this.todoRepository = todoRepository;
	}

	public List<Todo> getTodos() {
		return todoRepository.findAll().stream().map(TypeMapper::map).toList();
	}

	@Override
	public Todo findTodo(long id) throws TodoNotFoundException {
		return todoRepository.findById(id).map(TypeMapper::map)
				.orElseThrow(() -> new TodoNotFoundException("Todo with id " + id + " not found"));
	}

	@Override
	public long addTodo(String title, XMLGregorianCalendar dueDate) {
		TodoEntity todo = new TodoEntity(title, TypeMapper.map(dueDate));
		todo = todoRepository.save(todo);
		log.info("Todo with id " + todo.getId() + " added");
		return todo.getId();
	}

	@Override
	public void updateTodo(Todo todo) throws TodoNotFoundException {
		if (!todoRepository.existsById(todo.getId()))
			throw new TodoNotFoundException("Todo with id " + todo.getId() + " not found");
		todoRepository.save(TypeMapper.map(todo));
		log.info("Todo with id " + todo.getId() + " updated");
	}

	@Override
	public void removeTodo(long id) {
		todoRepository.findById(id).ifPresent(todoRepository::delete);
		log.info("Todo with id " + id + " removed");
	}
}
